// In this example the coroutine reads row-by-row text from a file. When a row is read, the coroutine 
// prints it and suspends. The human operator reads the text and resumes the reading or ends the work.
#include <coroutine>
class ReturnType
{
// The ReturnType class (the class name may be, of course, any) must contain:
// 1. Nested definition of a class that presents the coroutine promise (this is not the promise used in multithreading but its function is similar:
// the coroutine submits the results and exceptions through the promise object. Traditionally, the promise is defined as "struct promise_type".
public:
	struct promise_type
	{ // all those functions are mandatory
		auto get_return_object() { return ReturnType{ coroutine_handle<promise_type>::from_promise(*this) }; }
		// Called to create and initialize the coroutine inteface. Static method coroutine_handle<promise_type>::from_promise creates from the current promise
		// the handle, this handle is used as argument of the ReturnType constructor
		auto initial_suspend() { return suspend_always(); }
		// If this function returns object from standard class suspend_always, the coroutine is suspended at the very beginning and starts to wait for the
		// resume instruction (starting lazily). If the return value is of class suspend_never, the coroutine runs until the first susoending point (starting
		// eagerly)
		void unhandled_exception() { terminate(); }
		// How to deal with exceptions that the coroutine does not handle. Complicated problem, here we simply terminate the program
		void return_void() { }
		// In this example it must be empty, because our coroutine neither return any value nor yields results
		auto final_suspend() noexcept { return suspend_always(); }
		// Similar to initial_suspend: defines wether the coroutine must be suspended at the end.
	};
// 2. The coroutine handle, this must be an object of class coroutine_handle<promise_type>
	coroutine_handle<promise_type> handle;
// 3. Constructor
	ReturnType(coroutine_handle<promise_type> h) : handle(h) {}
// 4. Destructor
	~ReturnType() { handle.destroy(); } // method destroy() from class coroutine_handle<promise_type> destroys coroutine
// 5. Function resume()
	bool resume() const
	{
		if (!handle || handle.done())
		{  // if the coroutine has exited, method done() from class coroutine_handle<promise_type> return false
			return false;
		}
		handle.resume(); // method resume() from class coroutine_handle<promise_type> resumes the coroutine
		return !handle.done();
	}
};

ReturnTupe coro_reader(fstream & f)
{
	cout << "started" << endl;
	char line[256];
	while (true)
	{
		f.getline(line, 256);
		if (f.eof())
		{
			cout << "No more data" << endl;
			break;
		}
		cout << "has read text: " << line << endl;
		co_await suspend_always{};// Suspend point
								  // co_await is an operator, its operand is an object specifying the behavior of suspension
								  // Here we use nameless object from standard class suspend_always
								  // Object from class suspend_always switches the control back to caller
	}
	cout << "ended" << endl; // no return expression here, the control is switched back to caller
}

void Test()
{
	fstream File;
	File.open("C:\\Temp\\sometext.txt", fstream::in);
	if (!File.good())
	{
		cout << "Failed to open file" << endl;
		return;  
	}
	ReturnType task = coro_reader(File); // Initalize and launch coro_reader
	while (true)  
	{
		cout << "Press a key to continue, ESC to stop" << endl;
		if (_getch() == 27)
		{
			break;  // ESC - stop reading from file
		}
		if (!task.resume())  // When coro_reader has suspended its work, resume it after keystroke
		{
			break;  // coroutine has exited
		}
	}
	File.close();
}

// Comments:
// auto initial_suspend() { return suspend_always(); } means that the coroutine stops at the beginning. 
// The caller runs until the point where resume() is called. There it stops and the coroutine starts to
// run. It runs until the suspend point. 
// auto initial_suspend() { return suspend_never(); } means that the coroutine starts to run immediately
// and the caller is blocked. The coroutine stops at the suspend point. After that the caller ismunblocked
// and runs until the point where resume() is called.

